<?php
/**
 * Created by vm.pl
 * User: Rafał Ignaszewski (rafal@vm.pl)
 * Date: 29.11.13 11:17
 */

namespace VM\ApiBundle\Services;


use Doctrine\ORM\EntityManager;
use FOS\UserBundle\Model\UserManager;
use FOS\UserBundle\Util\TokenGenerator;
use FOS\UserBundle\Util\UserManipulator;
use Symfony\Component\Validator\Validator;
use VM\ApiBundle\Entity\Application;
use VM\ApiBundle\Entity\User;
use VM\ApiBundle\Entity\UserToken;
use VM\ApiBundle\Exceptions\ApiException;

class ApiUserService
{
    /**
     * @var EntityManager
     */
    protected $EntityManager;

    /**
     * @var \FOS\UserBundle\Model\UserManager
     */
    protected $UserManager;

    /**
     * @var PasswordEncoder
     */
    protected $PasswordEncoder;

    /**
     * @var UserManipulator
     */
    protected $UserManipulator;

    /**
     * @var Validator
     */
    protected $Validator;

    /**
     * @var TokenGenerator
     */
    protected $TokenGenerator;


    /**
     * @var int
     */
    public $tokenValidationTime = 2700;

    /**
     * Ważność tokenu dla urządzeń nie anonimowych = 3600 * 24 * 30
     *
     * @var int
     */
    public $tokenDeviceValidationTime = 2592000;


    public function __construct(EntityManager $EntityManager, UserManager $UserManager, PasswordEncoder $PasswordEncoder, UserManipulator $UserManipulator, Validator $Validator, TokenGenerator $TokenGenerator)
    {
        $this->EntityManager = $EntityManager;
        $this->UserManager = $UserManager;
        $this->PasswordEncoder = $PasswordEncoder;
        $this->UserManipulator = $UserManipulator;
        $this->Validator = $Validator;
        $this->TokenGenerator = $TokenGenerator;
    }


    /**
     * @param $email
     * @param $password
     * @param string $deviceID
     * @return array
     * @throws \VM\ApiBundle\Exceptions\ApiException
     */
    public function login($email, $password, $deviceID = '')
    {
        $User = $this->getUserByEmail($email);

        if($User->getConfirmationToken() !== null)
        {
            throw new ApiException('User account not confirmed yet', 20007);
        }

        if(!$User->isEnabled())
        {
            throw new ApiException('User account is not enabled', 20007);
        }

        if(!$User->isAccountNonLocked())
        {
            throw new ApiException('User account is not enabled', 20008);
        }

        if(!$this->PasswordEncoder->isPasswordValid($User->getPassword(), $password, $User->getSalt()))
        {
            throw new ApiException('incorrect login', 20001);
        }

        $tokens = $User->getTokens();
        if($tokens->count() >= $User->getMaxAllowedTokens())
        {
            $this->removeFirstNoDeviceToken($User);
        }

        $User->setLastLogin(new \DateTime());
        $UserToken = $this->createToken($User, $deviceID);

        $this->EntityManager->flush();

        return array('user_id' => $User->getId(), 'token' => $UserToken->getToken());
    }


    /**
     * Check if user token is valid
     *
     * @param $user_id
     * @param $token
     * @return array
     * @throws \VM\ApiBundle\Exceptions\ApiException
     */
    public function check_token($user_id, $token)
    {
        $User = $this->UserManager->findUserBy(array('id' => $user_id));
        if(empty($User))
        {
            throw new ApiException('User not exists', 20000);
        }

        $UserToken = $this->getTokenIfNotExpire($User, $token);
        if($UserToken instanceof UserToken)
        {
            $device = $UserToken->getDevice();
            $UserToken->setExpireAt($this->addDateValidTokenInterval(new \DateTime(), !empty($device)));
            $this->EntityManager->flush();
            return array('token_valid' => true);
        }
        else
        {
            throw new ApiException('Token valid is expired', 20002);
        }

    }


    /**
     * Logout user
     *
     * @param $user_id
     * @param $token
     * @return array
     * @throws \VM\ApiBundle\Exceptions\ApiException
     */
    public function logout($user_id, $token)
    {
        $User = $this->UserManager->findUserBy(array('id' => $user_id));
        if(empty($User))
        {
            throw new ApiException('User not exists', 20000);
        }

        $UserToken = $this->getTokenIfNotExpire($User, $token);
        if($UserToken instanceof UserToken)
        {
            $this->EntityManager->remove($UserToken);
            $this->EntityManager->flush();
            return array('logout' => true);
        }
        else
        {
            throw new ApiException('Token valid is expired', 20002);
        }

    }


    /**
     * @param $email
     * @param $password
     * @param $name
     * @param $surname
     * @return array
     * @throws \VM\ApiBundle\Exceptions\ApiException
     */
    public function register($email, $password, $name, $surname)
    {
        $User = $this->UserManager->createUser();
        $User->setUsername($email);
        $User->setEmail($email);
        $User->setPlainPassword($password);
        $User->setEnabled(false);
        $User->setSuperAdmin(false);

        $User->setConfirmationToken($this->TokenGenerator->generateToken());

        $this->UserManager->updateUser($User, false);

        $User->setName($name);
        $User->setSurname($surname);


        $this->validateUser($User);

        $this->EntityManager->flush();
        return array('user_id' => $User->getId(), 'confirmation_token' => $User->getConfirmationToken());
    }


    /**
     * Regiistration confirm action
     *
     * @param $user_id
     * @param $confirmation_token
     * @return array
     * @throws \VM\ApiBundle\Exceptions\ApiException
     */
    public function registration_confirm($user_id, $confirmation_token)
    {
        $User = $this->getUser($user_id);

        if(!$User->isAccountNonLocked())
        {
            throw new ApiException('User account is not enabled', 20008);
        }


        if($User->isEnabled())
        {
            throw new ApiException('User account is activated', 20006);
        }


        if($User->getConfirmationToken() != $confirmation_token)
        {
            throw new ApiException('Incorrect user confirmation token', 20005);
        }

        $User->setConfirmationToken(null);
        $User->setEnabled(true);

        $this->EntityManager->flush();
        return array('success' => true);
    }


    /**
     * @param $user_id
     * @param $token
     * @param $data
     * @return array
     * @throws \VM\ApiBundle\Exceptions\ApiException
     */
    public function update($user_id, $token, $data)
    {
        $User = $this->getUser($user_id);

        if(!$User->isAccountNonLocked())
        {
            throw new ApiException('User account is not enabled', 20008);
        }

        if(!$User->isEnabled())
        {
            throw new ApiException('User account is not activated', 20007);
        }

        $UserToken = $this->getTokenIfNotExpire($User, $token);
        if($UserToken === false)
        {
            throw new ApiException('User token is expired', 20002);
        }

        if(empty($data))
        {
            throw new ApiException('No data to update', 20009);
        }

        if(isset($data['name']))
        {
            $User->setName($data['name']);
        }

        if(isset($data['surname']))
        {
            $User->setSurname($data['surname']);
        }

        if(isset($data['password']))
        {
            $User->setPlainPassword($data['password']);
        }

        $this->UserManager->updateUser($User, false);

        $this->validateUser($User);

        $this->EntityManager->flush();

        return array('success' => true);
    }


    /**
     * @param $email
     * @return array
     * @throws \VM\ApiBundle\Exceptions\ApiException
     */
    public function generate_password_recovery_token($email)
    {
        $User = $this->getUserByEmail($email);

        $DateTime = new \DateTime();
        $passwordRecoveryToken = hash('sha512', sha1($email . $DateTime->format('YmdHis')));
        $User->setPasswordRecoveryToken($passwordRecoveryToken);
        $this->EntityManager->flush();

        return array("password_recovery_token" => $passwordRecoveryToken);
    }


    /**
     * @param $email
     * @param $password_recovery_token
     * @param $password
     * @return array
     * @throws \VM\ApiBundle\Exceptions\ApiException
     */
    public function change_password($email, $password_recovery_token, $password)
    {
        $User = $this->getUserByEmail($email);

        if($User->getPasswordRecoveryToken() !== $password_recovery_token)
        {
            throw new ApiException('Wrong password recovery token', 20010);
        }

        $User->setPlainPassword($password);
        $User->setPasswordRecoveryToken(null);
        $this->UserManager->updateUser($User, false);
        $this->EntityManager->flush();
        return array('success' => true);
    }

    public function check_password($user_id, $token, $password)
    {
        $User = $this->getUser($user_id);

        if(!$User->isAccountNonLocked())
        {
            throw new ApiException('User account is not enabled', 20008);
        }

        if(!$User->isEnabled())
        {
            throw new ApiException('User account is not activated', 20007);
        }

        $UserToken = $this->getTokenIfNotExpire($User, $token);
        if($UserToken === false)
        {
            throw new ApiException('User token is expired', 20002);
        }

        return array('success' => $this->PasswordEncoder->isPasswordValid($User->getPassword(),$password,$User->getSalt()));
    }


    /**
     * @param $user_id
     * @param $token
     * @return array
     * @throws \VM\ApiBundle\Exceptions\ApiException
     */
    public function get($user_id, $token)
    {
        $User = $this->getUser($user_id);

        $UserToken = $this->getTokenIfNotExpire($User, $token);
        if($UserToken === false)
        {
            throw new ApiException('User token is expired', 20002);
        }

        $lastLogin = $User->getLastLogin();
        return array(
            'user_id' => $User->getId(),
            'email' => $User->getEmail(),
            'name' => $User->getName(),
            'surname' => $User->getSurname(),
            'last_login' => (empty($lastLogin)) ? null : $lastLogin->format("Y-m-d H:i:s")
        );
    }


    /**
     * @param $users
     * @return array
     */
    public function get_names($users)
    {
        $resp = array('users' => array());

        foreach($users as $userId)
        {
            $User = $this->getUser($userId);
            $resp['users'][] = array(
                'user_id' => $User->getId(),
                'name' => $User->getName(),
                'surname' => $User->getSurname(),
                'email' => $User->getEmail(),
            );
        }

        return $resp;
    }


    /**
     * Get user roles in application
     *
     * @param $user_id
     * @param $token
     * @param $application
     * @return array
     * @throws \VM\ApiBundle\Exceptions\ApiException
     */
    public function get_roles($user_id, $token, $application)
    {
        $User = $this->getUser($user_id);

        $UserToken = $this->getTokenIfNotExpire($User, $token);
        if($UserToken === false)
        {
            throw new ApiException('User token is expired', 20002);
        }

        $Application = $this->getApplication($application);

        $userApplicationRoles = $User->getAppRoles();
        $roles = array();
        foreach($userApplicationRoles as $ApplicationRole)
        {
            if($ApplicationRole->getAppId() == $Application->getId())
            {
                $roles[] = $ApplicationRole->getSymbol();
            }
        }

        return array('roles' => $roles);
    }


    /**
     * Set user roles in application - remove all exists roles and sets new
     *
     * @param $user_id
     * @param $token
     * @param $application
     * @param $roles
     * @return array
     * @throws \VM\ApiBundle\Exceptions\ApiException
     */
    public function set_roles($user_id, $token, $application, $roles)
    {
        $User = $this->getUser($user_id);

        $UserToken = $this->getTokenIfNotExpire($User, $token);
        if($UserToken === false)
        {
            throw new ApiException('User token is expired', 20002);
        }

        $Application = $this->getApplication($application);

        $userApplicationRoles = $User->getAppRoles();
        foreach($userApplicationRoles as $ApplicationRole)
        {
            if($ApplicationRole->getAppId() == $Application->getId())
            {
                $User->removeAppRole($ApplicationRole);
            }
        }

        $applicationRoles = $this->EntityManager->getRepository('VMApiBundle:ApplicationRole')->findBy(array('appId'=> $Application->getId(), 'symbol' => $roles));
        foreach($applicationRoles as $ApplicationRole)
        {
            $User->addAppRole($ApplicationRole);
            $ApplicationRole->addUser($User);
        }

        $this->EntityManager->flush();

        return array('success' => true);
    }


    /**
     * @param $user_id
     * @return array
     */
    public function app_delete($user_id)
    {
        $User = $this->getUser($user_id);

        foreach($User->getAppRoles() as $ApplicationRole)
        {
            $User->getAppRoles()->removeElement($ApplicationRole);
        }

        $this->EntityManager->remove($User);
        $this->EntityManager->flush();

        return array('success' => true);
    }


    /**
     * @param $user_id
     * @param $data
     * @return array
     * @throws \VM\ApiBundle\Exceptions\ApiException
     */
    public function app_update($user_id, $data)
    {
        $User = $this->getUser($user_id);

        if(empty($data))
        {
            throw new ApiException('No data to update', 20009);
        }

        if(isset($data['name']))
        {
            $User->setName($data['name']);
        }

        if(isset($data['surname']))
        {
            $User->setSurname($data['surname']);
        }

        $this->validateUser($User);
        $this->EntityManager->flush();

        return array('success' => true);
    }

    /**
     * use only FacebookController
     *
     * @param $facebookUserProfile
     * @return array
     * @throws \VM\ApiBundle\Exceptions\ApiException
     * @throws \Exception
     */
    public function login_by_facebook($facebookUserProfile)
    {
        if(!isset($facebookUserProfile['id']))
        {
            throw new ApiException('Not set require param: id');
        }

        $User = $this->getUserByExternalId($facebookUserProfile['id'], false);
        if(empty($User))
        {
            if(isset($facebookUserProfile['email']))
            {
                $User = $this->UserManager->findUserByEmail($facebookUserProfile['email']);
                if(!empty($User))
                {
                    $User->setExternalId($facebookUserProfile['id']);
                }
            }
            if(empty($User))
            {
                $User = $this->createUserFromFacebookData($facebookUserProfile);
            }
        }

        if($User->getConfirmationToken() !== null)
        {
            throw new \Exception('User account not confirmed yet', 20007);
        }

        if(!$User->isEnabled())
        {
            throw new \Exception('User account is not enabled', 20007);
        }

        if(!$User->isAccountNonLocked())
        {
            throw new \Exception('User account is not enabled', 20008);
        }

        $tokens = $User->getTokens();
        if($tokens && $tokens->count() >= $User->getMaxAllowedTokens())
        {
            $this->removeFirstNoDeviceToken($User);
        }

        $User->setLastLogin(new \DateTime());
        $UserToken = $this->createToken($User, '');

        $this->EntityManager->flush();
        $this->EntityManager->refresh($User);

        return array('user_id' => $User->getId(), 'token' => $UserToken->getToken());
    }


    /**
     * @param User $User
     * @param string $device
     * @return UserToken
     */
    protected function createToken(User $User, $device = '')
    {
        if(!empty($device))
        {
            $Token = $this->findUserTokenByDevice($User->getId(), $device);
        }

        if(empty($Token))
        {
            $Token = new UserToken();
            $Token->setUser($User);
            if(!empty($device))
            {
                $Token->setDevice($device);
            }
            $this->EntityManager->persist($Token);
        }
        $DateTime = new \DateTime();
        $Token->setExpireAt($this->addDateValidTokenInterval($DateTime, !empty($device)));
        $Token->setToken($this->generateToken($Token));


        return $Token;
    }


    /**
     * @param UserToken $Token
     * @return string
     */
    protected function generateToken(UserToken $Token)
    {
        $DateTime = new \DateTime();
        $string = mt_rand(1, 100000000) . $Token->getUser()->getEmail() + $DateTime->format("YmdHis");
        return sha1($string);
    }


    /**
     * @param UserToken $UserToken
     * @return bool
     */
    protected function isTokenExpired(UserToken $UserToken)
    {
        $CurrentTime = new \DateTime();
        $ExpireAt = clone($UserToken->getExpireAt());
//        $this->addDateValidTokenInterval($ExpireAt);
//        var_dump($ExpireAt, $CurrentTime);
        return $ExpireAt < $CurrentTime;
    }


    /**
     * @param \DateTime $DateTime
     * @param $device
     * @return \DateTime
     */
    protected function addDateValidTokenInterval(\DateTime $DateTime, $device = '')
    {
        $seconds = ($device) ? $this->tokenDeviceValidationTime : $this->tokenValidationTime;
        return $DateTime->add(new \DateInterval('PT' . $seconds . 'S'));
    }


    /**
     * @param $userId
     * @return \VM\ApiBundle\Entity\User
     * @throws \VM\ApiBundle\Exceptions\ApiException
     */
    protected function getUser($userId)
    {
        $User = $this->EntityManager->getRepository('VMApiBundle:User')->find($userId);
        if(empty($User))
        {
            throw new ApiException('User not exist', 20004);
        }
        return $User;
    }


    /**
     * @param $email
     * @return \VM\ApiBundle\Entity\User
     * @throws \VM\ApiBundle\Exceptions\ApiException
     */
    protected function getUserByEmail($email)
    {
        $User = $this->UserManager->findUserByEmail($email);
        if(empty($User))
        {
            throw new ApiException('User not exist', 20000);
        }
        return $User;
    }

    /**
     * @param $externalId
     * @param bool $error
     * @return mixed
     * @throws \VM\ApiBundle\Exceptions\ApiException
     */
    protected function getUserByExternalId($externalId, $error = true)
    {
        $User = $this->UserManager->findUserBy(array('external_id' => $externalId));
        if(empty($User) && $error)
        {
            throw new ApiException('User not exist', 20000);
        }
        return $User;
    }

    /**
     * @param $appSymbol
     * @return Application
     * @throws \VM\ApiBundle\Exceptions\ApiException
     */
    protected function getApplication($appSymbol)
    {
        $Application = $this->EntityManager->getRepository('VMApiBundle:Application')->findOneBySymbol($appSymbol);
        if(empty($Application))
        {
            throw new ApiException('Application not exist', 20011);
        }
        return $Application;
    }



    /**
     * @param User $User
     * @param $token
     * @return bool
     */
    protected function getTokenIfNotExpire(User $User, $token)
    {
        foreach($User->getTokens() as $UserToken)
        {
            if($UserToken->getToken() === $token)
            {
                if(!$this->isTokenExpired($UserToken))
                {
                    return $UserToken;
                    break;
                }
            }
        }
        return false;
    }


    /**
     * @param $userId
     * @param $device
     * @return UserToken|bool
     */
    protected function findUserTokenByDevice($userId, $device)
    {
        $Token = $this->EntityManager->getRepository('VMApiBundle:UserToken')->findOneBy(array('userId' => $userId, 'device' => $device));
        if($Token instanceof UserToken)
        {
            return $Token;
        }
        return false;
    }


    /**
     * @param User $User
     * @return bool
     */
    protected function removeFirstNoDeviceToken(User $User)
    {
        $tokens = $User->getTokens();
        $removed = false;

        foreach($tokens as $UserToken)
        {
            $device = $UserToken->getDevice();
            if(empty($device))
            {
                $this->EntityManager->remove($UserToken);
                $removed = true;
                break;
            }
        }

        return $removed;
    }

    /**
     * @param $facebookUserProfile
     * @return User
     */
    protected function createUserFromFacebookData($facebookUserProfile)
    {
        $email = (isset($facebookUserProfile['email'])) ? $facebookUserProfile['email'] : $facebookUserProfile['id'].'@facebook.com';

        $User = new User();

        $User->setExternalId($facebookUserProfile['id']);
        $User->setName((isset($facebookUserProfile['first_name'])) ? $facebookUserProfile['first_name'] : '');
        $User->setSurname((isset($facebookUserProfile['last_name'])) ? $facebookUserProfile['last_name'] : '');
        $User->setEmail($email);
        $User->setUsername($email);
        $User->setEmailCanonical($email);
        $User->setEnabled(true);
        $User->setPassword($email);

        $this->EntityManager->persist($User);

        return $User;
    }


    /**
     * @param User $User
     * @throws \VM\ApiBundle\Exceptions\ApiException
     */
    protected function validateUser(User $User)
    {
        $errors = $this->Validator->validate($User);
        if(count($errors) > 0)
        {
            throw new ApiException((string)$this->__formatValidationErrors($errors), 20003);
        }
    }


    /**
     * @param $errors
     * @return string
     */
    protected function __formatValidationErrors($errors)
    {
        $err = array();
        foreach($errors as $error)
        {
            $err[] = $error->getMessage();
        }
        return implode('|', $err);
    }
} 